package lendables;
import java.util.Vector; // Use this only for the test driver

/**
 * A concrete realization of Lendable.
 * 
 * This class implements Lendable's two abstract methods 1. checkSelfOut 2.
 * getLateFee
 * 
 * One new instance variable and one new accessor method are added 1. my_artist
 * 2. getArtist()
 */
public class CD extends Lendable {
	public static final int DAYS_TO_BORROW = 7;
	public static final double LATE_DAY_FEE = 2.50;
	
	private String my_artist;

	/**
	 * Initialize a CD object
	 * 
	 * @param callNumber
	 *            The unique identifier
	 * @param title
	 *            The title of this Lendable
	 * @param artist
	 *            The Artist of this CD
	 */
	public CD(String callNumber, String title, String artist) {
		super(title, callNumber, DAYS_TO_BORROW, LATE_DAY_FEE);
		my_artist = artist;
	}

	/**
	 * Find out the artist of this CD.
	 * 
	 * @return The artist name, an individual or group.
	 */
	public String getArtist() {
		return my_artist;
	}

	/**
	 * This method is called during println messages.
	 * 
	 * @return a string representation of this item
	 */
	@Override
	public String toString() {
		return super.toString();
	}

	public static void main(String[] args) {
		// A Test Tester for class CD
		Vector<Lendable> cds = new Vector<Lendable>();

		Lendable aCD = new CD("1", "Empty", "Jackson Browne");
		System.out.print("Error? ");
		aCD.checkSelfIn(); // Can't do

		System.out.println(" available true? " + aCD.isAvailable());
		System.out.println("  overdue false? " + aCD.isOverdue());
		System.out.println("  borrower null? " + aCD.getBorrowerID());
		aCD.checkSelfOut("KIM");
		System.out.println("available false? " + aCD.isAvailable());
		System.out.println("  overdue false? " + aCD.isOverdue());
		System.out.println("  current state: " + aCD.toString());
		System.out.println("   Late fee 0.0? " + aCD.getLateFee());
		System.out.println("   borrower KIM? " + aCD.getBorrowerID());
		System.out.println("        call# 1? " + aCD.getCallNumber());
		System.out.println("    title Empty? " + aCD.getTitle());
		System.out.println();
		DayCounter today = new DayCounter();
		System.out.println("due date " + DAYS_TO_BORROW + " days from "
				+ today + "? " + aCD.getDueDate());

		// Build a small collection of books
		cds.add(new CD("2", "Lives in the Balance", "Jackson Browne"));
		cds.add(new CD("3", "Back in the Highlife Again", "Peter Gabriel"));
		cds.add(new CD("4", "Revolver", "Beatles"));

		System.out.println("\nCurrent vector of cds: ");
		for (Lendable cd : cds) {
			System.out.println(cd);
		}

		cds.get(0).checkSelfIn(); // Can't do ERROR
		cds.get(0).checkSelfOut("KIM");
		cds.get(1).checkSelfOut("JO");
		cds.get(2).checkSelfOut("JO");
		cds.get(1).checkSelfIn();

		System.out.println("\nVector of cds after some check-outs and -ins: ");
		for (Lendable cd : cds) {
			System.out.println(cd);
		}
	}
}
/*
Expected output from test driver

Error? *ERROR* 1 is not checked out
 available true? true
  overdue false? false
  borrower null? null
available false? false
  overdue false? false
  current state: CD: 1 'Empty': due 2008/09/07, KIM
   Late fee 0.0? 0.0
   borrower KIM? KIM
        call# 1? 1
    title Empty? Empty

due date 7 days from 2008/08/31? 2008/09/07

Current vector of cds: 
CD: 2 'Lives in the Balance': available
CD: 3 'Back in the Highlife Again': available
CD: 4 'Revolver': available
*ERROR* 2 is not checked out

Vector of cds after some check-outs and -ins: 
CD: 2 'Lives in the Balance': due 2008/09/07, KIM
CD: 3 'Back in the Highlife Again': available
CD: 4 'Revolver': due 2008/09/07, JO

 */