package lendables;

/**
 * A concrete realization of Lendable.
 * 
 * This class implements Lendable's two abstract methods 1. checkSelfOut 2.
 * getLateFee
 * 
 * One new instance variable and one new accessor method are added 1. my_artist
 * 2. getArtist()
 */
public class Book extends Lendable {
	public static final int DAYS_TO_BORROW = 14;
	public static final double LATE_DAY_FEE = 0.50;

	private String my_author;

	/**
	 * Initialize a Book object.
	 * 
	 * @param callNumber
	 *            The unique identifier.
	 * @param title
	 *            The title of this Lendable.
	 * @param artist
	 *            The Author of this book.
	 */
	public Book(String callNumber, String title, String author) {
		super(title, callNumber, DAYS_TO_BORROW, LATE_DAY_FEE);
		my_author = author;
	}

	/**
	 * Find out the author of this Book.
	 * 
	 * @return The author's name.
	 */
	public String getAuthor() {
		return my_author;
	}

	/**
	 * This method is called during println messages.
	 * 
	 * @return a string representation of this item
	 */
	@Override
	public String toString() {
		return super.toString();
	}

	public static void main(String[] args) {
		Book aBook = new Book("QA76.1", "Data Structures", "Mike Berman");

		// Try to check in a book that is not checked out
		System.out.print("Error? ");
		aBook.checkSelfIn(); // This is not allowed, but it could happen

		System.out.println(" available true? " + aBook.isAvailable());
		System.out.println("  overdue false? " + aBook.isOverdue());
		System.out.println("  borrower null? " + aBook.getBorrowerID());
		aBook.checkSelfOut("KIM");
		System.out.println("available false? " + aBook.isAvailable());
		System.out.println("  overdue false? " + aBook.isOverdue());
		System.out.println("  current state: " + aBook.toString());
		System.out.println("   Late fee 0.0? " + aBook.getLateFee());
		System.out.println("   borrower KIM? " + aBook.getBorrowerID());
		System.out.println("   call# QA76.1? " + aBook.getCallNumber());
		System.out.println("Data Structures? " + aBook.getTitle());

		// Build a small array of Book objects
		Book[] books = new Book[4];

		books[0] = aBook; // aBook was constructed above � it is currently
		// checked
		// out.
		books[1] = new Book("QA76.2", "Karel the Robot", "Rich Pattis");
		books[2] = new Book("QA76.3", "Oh Pascal", "Mike Clancy");
		books[3] = new Book("QA76.4", "A C.S. Tapestry", "Owen Astrachan");

		// Show the state of all Books in the List
		System.out.println("\nCurrent List of books: ");
		for (Book b : books) {
			System.out.println(b.toString());
		}

		books[0].checkSelfIn();
		books[1].checkSelfOut("JO");
		books[2].checkSelfOut("JO");
		books[3].checkSelfOut("JO");

		// Show the state of all Books in the List
		System.out.println("\nList of books after other check-outs and -ins: ");
		for (Book b : books) {
			System.out.println(b.toString());
		}

		DayCounter thirtyDaysAgo = new DayCounter();
		thirtyDaysAgo.adjustDaysBy(-30);
		Book overdue = new Book("Q", "Overdue", "Late fee = 30 * 0.50");
		overdue.checkSelfOut("Kim");
		System.out.println("Correct due date: " + overdue.getDueDate());
		System.out.println("false? " + overdue.isOverdue());
		System.out.println("0.00? " + overdue.getLateFee());
		overdue.setDueDate(thirtyDaysAgo);
		System.out.println("Modified due date: " + overdue.getDueDate());
		System.out.println("true? " + overdue.isOverdue());
		System.out.println("15.00? " + overdue.getLateFee());
	}
}

/*
Expected output from test driver

Error? *ERROR* QA76.1 is not checked out
 available true? true
  overdue false? false
  borrower null? null
available false? false
  overdue false? false
  current state: Book: QA76.1 'Data Structures': due 2008/09/14, KIM
   Late fee 0.0? 0.0
   borrower KIM? KIM
   call# QA76.1? QA76.1
Data Structures? Data Structures

Current List of books: 
Book: QA76.1 'Data Structures': due 2008/09/14, KIM
Book: QA76.2 'Karel the Robot': available
Book: QA76.3 'Oh Pascal': available
Book: QA76.4 'A C.S. Tapestry': available

List of books after other check-outs and -ins: 
Book: QA76.1 'Data Structures': available
Book: QA76.2 'Karel the Robot': due 2008/09/14, JO
Book: QA76.3 'Oh Pascal': due 2008/09/14, JO
Book: QA76.4 'A C.S. Tapestry': due 2008/09/14, JO
Correct due date: 2008/09/14
false? false
0.00? 0.0
Modified due date: 2008/08/01
true? true
15.00? 15.0

*/