package examples;

import java.io.*;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;

/**
 * This class represents a pet survey, keeping track of the number of votes cast
 * for a set of pets. The results are stored in a file.
 * 
 * Modified to use a collection.
 * 
 * @author CS209
 * 
 */
public class PetSurvey3 {

	private String filename;
	private int totalVotes = 0;

	Map<String, Integer> petToVotes;
	
	public PetSurvey3(String fn) {
		this.filename = fn;
		petToVotes = new HashMap<String, Integer>();
		importResults();
	}

	/**
	 * Read the survey data from the file
	 */
	private void importResults() {
		try {
			BufferedReader input = new BufferedReader(new FileReader(filename));

			String line;
			int i = 0;
			while ((line = input.readLine()) != null) {
				// line format: <animal> <numVotes>
				String data[] = line.split(" ");

				String animal = data[0];
				int numVotes = Integer.parseInt(data[1]);

				petToVotes.put(animal, numVotes);
				
				i++;
				totalVotes += numVotes;
			}

			input.close();
		} catch (FileNotFoundException e) {
			System.out.println(filename
					+ ", containing the survey results, not found");
			e.printStackTrace();
		} catch (IOException e) {
			System.out
					.println("Something went wrong while importing survey results from"
							+ filename);
			e.printStackTrace();
		}

	}

	/**
	 * Store the current voting results into the file
	 */
	public void storeResults() {
		try {
			PrintWriter writer = new PrintWriter(filename);

			Set<String> keys = petToVotes.keySet();
			
			for (String animal: keys) {
				writer.println(animal + " " + petToVotes.get(animal));
			}

			writer.close();

		} catch (IOException e) {
			System.out.println("Error storing survey results to file "
					+ filename);
			e.printStackTrace();
		}

	}

	/**
	 * 
	 * @return the name of the file containing the survey results
	 */
	public String getFilename() {
		return filename;
	}

	/**
	 * 
	 * @return the array of Strings of animal names in the survey
	 */
	public String[] getAnimals() {
		Set<String> animals = petToVotes.keySet();
		String[] animalsArray = (String[]) animals.toArray();
		return animalsArray;
	}

	/**
	 * 
	 * @return the number of votes that have been cast.
	 */
	public int getTotalVotes() {
		return totalVotes;
	}

	/**
	 * Casts a vote for the animal
	 * 
	 * @param animal
	 */
	public void castVote(String animal) {
		Integer numVotes = petToVotes.get(animal);
		if( numVotes == null ) {
			petToVotes.put(animal, 1);
		} else {
			petToVotes.put(animal, numVotes+1);
		}
		totalVotes++;
	}

	/**
	 * Display the results from the survey in a nicely formatted way.
	 */
	public void displayResults() {
		System.out.printf("%-10s%7s%13s\n", "Animal", "Votes", "Percentage");
		System.out.println("------------------------------");
		
		Set<String> keys = petToVotes.keySet();
		
		for( String animal : keys ) {
			int votes = petToVotes.get(animal);
			double pct = (double) votes/totalVotes * 100;
			System.out.printf("%-10s%7d%12.2f%%\n", animal, votes, pct);
		}
		System.out.println("Total votes cast: " + totalVotes);
	}

	/**
	 * @param args
	 *            not used in this program.
	 */
	public static void main(String[] args) {
		final String mySurveyFile = "petSurvey.dat";
		PetSurvey3 survey = new PetSurvey3(mySurveyFile);

		System.out.println("Current Results: ");
		survey.displayResults();

		// Allow User to Vote
		Scanner scanner = new Scanner(System.in);
		System.out
				.print("What animal do you want to vote for as your favorite? (dog, cat, bird, snake, fish, other): ");

		String animalVoted = scanner.nextLine();
		scanner.close();

		survey.castVote(animalVoted);

		// Display updated results
		System.out.println("Updated Results: ");
		survey.displayResults();

		survey.storeResults();
	}

}
