package lendables;
import java.util.GregorianCalendar;
import java.text.DecimalFormat;

/*
 * A DayCounter represents the current day set on the computer system clock. 
 *
 * You can advance that day into the future or into the past with the 
 * adjustDaysBy method. Two DayCounter objects can be compared to see 
 * how far apart they are with the daysFrom method.
 *
 * The month 1 means January and month 12 means December.
 *
 * @author Rick Mercer
 * @date   July-2000
 */
public class DayCounter {
	private GregorianCalendar current_date;
	private DecimalFormat yearFormatter = new DecimalFormat("0000");
	private DecimalFormat dayMonthFormatter = new DecimalFormat("00");

	/**
	 * The default constructor sets this DayCounter object to represent the
	 * current date of the computer's clock
	 */
	public DayCounter() {
		current_date = new GregorianCalendar();
	}

	/**
	 * Initialize a new DayCounter with the year, month, and data supplied as
	 * arguments, new DayCounter( 2002, 10, 8 ) is Oct-8 in the year 2002.
	 */
	public DayCounter(int year, int month, int day) {
		// Let this interface treat 1 as January.
		// GregorianCalendar needs 0 for January
		current_date = new GregorianCalendar(year, month - 1, day);
	}

	/**
	 * This toString method returns the date as year/month/date, no leading
	 * zeros, for example the 21st of August 2001 would be returned as
	 * 2001/8/21.
	 */
	@Override
	public String toString() {
		int year = current_date.get(GregorianCalendar.YEAR);
		int month = current_date.get(GregorianCalendar.MONTH);
		int day = current_date.get(GregorianCalendar.DATE);

		return "" + yearFormatter.format(year) + "/"
				+ dayMonthFormatter.format(month + 1) + "/"
				+ dayMonthFormatter.format(day);
	}

	/**
	 * Use this if you want a calendar date to represent a past (argument is
	 * negative) or future date.
	 */
	public void adjustDaysBy(int days) {
		current_date.add(GregorianCalendar.DATE, days);
	}

	/**
	 * Determine how many days apart this day is from the argument.
	 * 
	 * A positive return says it will take that many days to get to the future
	 * date. A negative return signifies that the date of the argument comes
	 * before this date.
	 * 
	 * @param The
	 *            day you are comparing this day to.
	 * @return The difference in days between the two dayCounter objects
	 */
	public int daysFrom(DayCounter arg) {
		int result = 0;

		// Construct two temporary GregorianCalendar objects.
		// This method does NOT We modify this or arg.
		int leftYear = current_date.get(GregorianCalendar.YEAR);
		int leftMonth = current_date.get(GregorianCalendar.MONTH);
		int leftDate = current_date.get(GregorianCalendar.DATE);
		GregorianCalendar left = new GregorianCalendar(leftYear, leftMonth,
				leftDate);

		int rightYear = arg.current_date.get(GregorianCalendar.YEAR);
		int rightMonth = arg.current_date.get(GregorianCalendar.MONTH);
		int rightDate = arg.current_date.get(GregorianCalendar.DATE);
		GregorianCalendar right = new GregorianCalendar(rightYear, rightMonth,
				rightDate);

		// Use the Gregorian Calenders's before, add, and after
		// methods to count the days to the future or to the past.
		while (left.before(right)) {
			result++;
			left.add(GregorianCalendar.DATE, 1);
		}

		// Assert: result will still be 0 if this date is greater than or equal
		// to the argument's date (left < right would have been false).

		// Now if the argument is before this date (right before left)
		// find out by how many days (or skip this loop when left was < right).
		while (left.after(right)) {
			result--;
			left.add(GregorianCalendar.DATE, -1);
		}

		// If the dates were equal, no loop executed and result is still 0;
		return result;
	}

	public static void main(String[] args) {
		DayCounter someDay = new DayCounter();
		System.out.println("a: " + someDay);
		someDay.adjustDaysBy(-2);
		System.out.println("b: " + someDay);
		someDay.adjustDaysBy(7);
		System.out.println("c: " + someDay);
		DayCounter today = new DayCounter();
		System.out.println("d: " + someDay.daysFrom(today));
		System.out.println("e: " + today.daysFrom(someDay));
	}
}

/*
Expected output from test driver (depends on the day)

a: 2008/08/31
b: 2008/08/29
c: 2008/09/05
d: -5
e: 5

*/