package lendables;

/**
 * An abstract class to capture the instance variables and methods that are
 * common to all objects that can be borrowed from a library.
 * 
 * This class also specifies what methods MUST be implemented by the subclasses.
 */
public abstract class Lendable {

	private String my_callNumber;
	private String my_title;
	private int my_borrowLength;

	private String my_borrowerID;
	private DayCounter my_dueDate;

	private double my_lateFee;
	private double my_firstDayLateFee;


	/**
	 * Create a generic Lendable object
	 * @param callNumber
	 * @param title
	 * @param borrowLength
	 * @param firstDayLateFee
	 * @param lateFee
	 * @param availability
	 */
	public Lendable(String callNumber, String title, int borrowLength,
			double firstDayLateFee, double lateFee,
			boolean availability) {
		super();
		this.my_callNumber = callNumber;
		this.my_title = title;
		this.my_borrowLength = borrowLength;
		this.my_firstDayLateFee = firstDayLateFee;
		this.my_lateFee = lateFee;
		
		// defaults:
		this.my_borrowerID = null;
		this.my_dueDate = null;
	}

	/**
	 * Create a generic Lendable object
	 * 
	 * First day late fee isn't different from late fee.
	 * 
	 * @param title
	 * @param callNumber
	 * @param borrowLength
	 * @param lateFee
	 * @param availability
	 */
	public Lendable(String title, String callNumber, int borrowLength,
			double lateFee, boolean availability) {
		this(callNumber, title, borrowLength, lateFee, lateFee, availability);
	}
	
	
	/**
	 * Create a generic Lendable object
	 * 
	 * First day late fee isn't different from lateFee.
	 * Available by default.
	 * 
	 * @param title
	 * @param callNumber
	 * @param borrowLength
	 * @param lateFee
	 */
	public Lendable(String title, String callNumber, int borrowLength,
			double lateFee) {
		this(callNumber, title, borrowLength, lateFee, lateFee, true);
	}
	
	/**
	 * Create a generic Lendable object
	 * 
	 * Available by default.
	 * 
	 * @param title
	 * @param callNumber
	 * @param borrowLength
	 * @param firstDayLateFee
	 * @param lateFee
	 */
	public Lendable(String title, String callNumber, int borrowLength,
			double firstDayLateFee, double lateFee) {
		this(callNumber, title, borrowLength, firstDayLateFee, lateFee, true);
	}

	/**
	 * Provide access to this object's call number.
	 * 
	 * @return this object's call number
	 */
	public String getCallNumber() {
		return my_callNumber;
	}

	/**
	 * Provide access to this object's title.
	 * 
	 * @return this object's title
	 */
	public String getTitle() {
		return my_title;
	}

	/**
	 * Let anyone know if this Lendable is available.
	 * 
	 * @return true if this Lendable is not currently borrowed.
	 */
	public boolean isAvailable() {
		return this.my_borrowerID == null;
	}

	/**
	 * Provide access to the borrower ID (could be null).
	 * 
	 * @return null if no one is borrowing this Lendable or return the String
	 *         that could be used to find the Borrower object.
	 */
	public String getBorrowerID() {
		return my_borrowerID;
	}

	/**
	 * Find out if a Lendable is overdue or not
	 * 
	 * @return true if this book was supposed to be returned before today or
	 *         false if the due date is past today.
	 */
	public boolean isOverdue() {
		// not even checked out
		if (this.isAvailable())
			return false;
		// This Lendable is checked out, so see if it is overdue
		else
			return daysLate() > 0;
	}

	/**
	 * Provide access to when a Lendable is due back in the Library
	 * 
	 * @return This Lendable's due date
	 */
	public DayCounter getDueDate() {
		return my_dueDate;
	}

	/**
	 * Provide a quick way see some important information about a Lendable.
	 * 
	 * @return The call number, title, and some information about the
	 *         availability (or due date) of this Lendable.
	 */
	@Override
	public String toString() {
		String result = this.getFormattedClass() + ": " + my_callNumber + " '" + my_title + "': ";
		if (this.isAvailable())
			result += "available";
		else
			result += "due " + my_dueDate.toString() + ", " + my_borrowerID;
		return result;
	}
	
	/**
	 * 
	 * @return this class formatted in the way we want
	 */
	private String getFormattedClass() {
		String s = this.getClass().toString();
		int periodPlace = s.lastIndexOf('.');
		s = s.substring(periodPlace+1);
		return s;
	}

	/**
	 * This method will correctly modify the state of any Lendable so it is
	 * considered returned and available for others to borrow.
	 * 
	 * @return true if the Lendable's state indicates it has been returned.
	 */
	public boolean checkSelfIn() {
		if (this.isAvailable()) {
			// this can not be checked out
			System.out.println("*ERROR* " + my_callNumber
					+ " is not checked out");
			return false;
		} else {
			my_dueDate = null;
			my_borrowerID = null;
			return true;
		}
	}

	/**
	 * Determine how many days late this Lendable is.
	 * 
	 * @return the number of days, which could be negative indicating the number
	 *         of days until the Lendable is due.
	 */
	public int daysLate() {
		int result = 0;
		if (!isAvailable()) {
			DayCounter today = new DayCounter();
			result = my_dueDate.daysFrom(today);
		}
		return result;
	}

	/**
	 * Only used for testing purposes.
	 */
	void setDueDate(DayCounter d) {
		my_dueDate = d;
	}

	/**
	 * Modify the state of this object so it is borrowed.
	 * 
	 * @param borrowerID
	 *            The String identification of the borrower.
	 */
	public void checkSelfOut(String borrowerID) {
		// Record who is borrowing this Lendable
		my_borrowerID = borrowerID;
		// Set the new due date by the correct number of days
		my_dueDate = new DayCounter();
		my_dueDate.adjustDaysBy(my_borrowLength);
	}

	/**
	 * Find out the dollar amount of this object's late fee.
	 * 
	 * @return The late fee if the Lendable were to be returned today
	 */
	public double getLateFee() {
		// not even checked out!
		if (isAvailable()) {
			return 0.00;
		} else {
			// A positive daysOverdue means that the due date is in the past.
			int daysOverdue = this.daysLate();
			if (daysOverdue <= 0)
				return 0.00;
			else
				return my_firstDayLateFee + (daysOverdue - 1) * my_lateFee;
		}
	}

}
