/**
 * A simple Java class that models a Chicken. The state of the chicken is its
 * name, height and weight.
 * 
 * @author Sara Sprenkle
 */
public class Chicken implements Comparable {

	// ------------ INSTANCE VARIABLES -------------------

    /** the chicken's name */
	protected String name;
	
    /** the height of the chicken in centimeters */
	protected int height;
    
    /** the weight of the chicken in pounds */
	protected double weight;
    
    /** true iff the chicken is a female */
    protected boolean isFemale;
    
    public static String DEFAULT_NAME = "BUBBA";

    public Chicken(String name, int height, double weight, boolean isFemale ) {
        this.name = name;
		this.height = height;
		this.weight = weight;
        this.isFemale = isFemale;
    }
    
    /**
    * Assumes Chickens are females
    */
    public Chicken(String name, int height, double weight) {
		this( name, height, weight, true );
	}
    
    /**
     * Default name is Bubba; Assumes Chickens are female
     */
    public Chicken( int height, double weight ) {
        this( DEFAULT_NAME, height, weight, true);
    }
    
    /**
     * Returns the chicken's name, weight, and height.
     */
    public String toString() {
        String rep = "Name: " + name + ", ";
        rep += "Weight: " + weight + ", ";
        rep += "Height: " + height + ", ";
        rep += isFemale ? "female" : "male";  // Java ternary operator; also available in C
        return rep;
    }

    /**
     * Determines if two Chickens are equivalent, based on their
     * name, height, weight, and gender.
     */
    public boolean equals(Object o) {
        
        // o can't be equal if it's not a Chicken
        if( ! (o instanceof Chicken) ) {
            return false;
        }
        
        Chicken other = (Chicken) o;
        
        if( ! other.getName().equals(this.getName() ) ) {
            return false;
        }
        
        if( other.getHeight() != this.getHeight() ) {
            return false;   
        }
        
        if( other.getWeight() != this.getWeight() ) {
            return false;   
        }
        
        if( this.isFemale() ) {
            return other.isFemale();
        } else {
            return ! other.isFemale();
        }
        
    }
    
    @Override
    /**
     * Compares Chickens based on their height.
     * If the other object is not a Chicken, then
     * that object is "greater than" this object.
     */
    public int compareTo(Object o) {
        // Chickens before others
        if( ! (o instanceof Chicken) ) {
            return -1;
        }
        
        Chicken other = (Chicken) o;
        if (height < other.getHeight() )
            return -1;
        if (height > other.getHeight())
            return 1;
        return 0;
    }
    
	//
	// ----------- GETTER METHODS ------------
	// (also Accessor methods)

	/**
	 * @return the height of the chicken, in centimeters
	 */
	public int getHeight() {
		return height;
	}

	public double getWeight() {
		return weight;
	}

	public String getName() {
		return name;
	}
    
    public boolean isFemale() {
        return isFemale;   
    }
    

	//
	// ------------- MUTATORS -----------
	//

	public void feed() {
		weight += .2;
		height += 1;
	}
	
	//
	// ------------- SETTERS ----------
	//

	/**
	 * @param n
	 *            the name of the chicken
	 */
	public void setName(String n) {
		name = n;
	}

	/**
	 * @param h
	 *            the height of the chicken, in cm
	 */
	public void setHeight(int h) {
		height = h;
	}
    
    /**
	 * @param w
	 *            the weight of the chicken, in pounds
	 */
	public void setWeight(double w) {
		weight = w;
	}
    
	/**
	 * @param args
	 *            the command-line arguments
	 */
	public static void main(String args[]) {
        Chicken one = new Chicken("Fred", 38, 2.0, false);
        Chicken two = new Chicken("Sallie Mae", 45, 3.0);
        Chicken oneFemale = new Chicken("Fred", 38, 2.0, true);
        
        System.out.println(one.getName() + " weighs " + one.getWeight() );
        one.feed();
        System.out.println(one.getName() + " ate, so he now weighs " + one.getWeight() );
        
        System.out.print(two.getName() + " is now ");
        two.setName("The Divine Miss Sallie Mae");
        System.out.println(two.getName());
        
        // test toString method
        System.out.println(one);
        
        System.out.println( one + " equal to " + two + "? " + one.equals(two) );
        System.out.println( one + " equal to " + oneFemale + "? " + one.equals(oneFemale) );
        System.out.println( two + " equal to " + two + "? " + two.equals(two) );

		System.out.println( "Fred vs. Miss Sallie Mae: " + one.compareTo(two) );
        System.out.println( "Miss Sallie Mae vs. Fred: " + two.compareTo(one) );
        System.out.println( "Fred vs. Female Fred: " + one.compareTo(oneFemale) );
        
	}

}